<?php
/* --------------------------------------------------------------
  Config.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Entities;

use \StyleEdit\Entities\Interfaces\ConfigInterface;

/**
 * Class Config
 *
 * @package StyleEdit\Entities
 */
class Config implements ConfigInterface
{
	/** @var array $jsonData */
	protected $jsonData;


	/**
	 * Config constructor.
	 *
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "name"-element with a non-empty
	 *                                   string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplate"-element with a
	 *                                   non-empty string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "colorPalette"-element
	 *                                   containing an array of string values
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "creationDate"-element with a
	 *                                   formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "modificationDate"-element with
	 *                                   a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplateModificationDate"-
	 *                                   element with a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "settings"-element
	 * @throws \InvalidArgumentException if $jsonData["settings"][]["icon"] is not a string
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "customStyles"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"] does not contain a valid
	 *                                   "entries"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "group"-element if $jsonData["settings"]["type"] is "display"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "position"-element if $jsonData["settings"]["type"] is "boxes"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   boolean value if $jsonData["settings"]["entries"][]["type"] is "switch"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   string value if $jsonData["settings"]["entries"][]["type"] is not "switch"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain an
	 *                                   "options" array if $jsonData["settings"]["entries"][]["type"] is "select"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"][]["options"] does not
	 *                                   contain string or array values
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "tags"-element containing an array of strings
	 * @throws \InvalidArgumentException if the given array $jsonData["customStyles"] does not contain arrays in the
	 *                                  format array("group" => "String", "content" => "String")
	 *
	 * @param array $jsonData
	 */
	public function __construct(array $jsonData)
	{
		$this->_validateJsonDataStructure($jsonData);

		$jsonData['isActive'] = false;
		$this->jsonData       = $jsonData;
	}


	/**
	 * Returns the name of the style configuration
	 *
	 * @return string
	 */
	public function getName()
	{
		return $this->jsonData['name'];
	}


	/**
	 * Returns the boilerplate's filename of the style configuration
	 *
	 * @return string
	 */
	public function getBoilerplateName()
	{
		return $this->jsonData['boilerplate'];
	}


	/**
	 * Returns the creation date of the style configuration as a DateTime object
	 *
	 * @return \DateTime
	 */
	public function getCreationDate()
	{
		return new \DateTime($this->jsonData['creationDate']);
	}


	/**
	 * Returns the modification date of the style configuration as a DateTime object
	 *
	 * @return \DateTime
	 */
	public function getModificationDate()
	{
		return new \DateTime($this->jsonData['modificationDate']);
	}
	
	
	/**
	 * Returns the boilerplate modification date of the style configuration as a DateTime object
	 *
	 * @return \DateTime
	 */
	public function getBoilerplateModificationDate()
	{
		if(!array_key_exists('boilerplateModificationDate', $this->jsonData))
		{
			return new \DateTime('1000-01-01 00:00:00');
		}
		
		return new \DateTime($this->jsonData['boilerplateModificationDate']);
	}


	/**
	 * Returns the complete configuration as an array
	 *
	 * @return array
	 */
	public function getJsonDataArray()
	{
		return $this->jsonData;
	}


	/**
	 * Updates the style configuration name
	 *
	 * @throws \InvalidArgumentException if $p_name is not a string
	 *
	 * @param string $p_name
	 *
	 * @return ConfigInterface Same instance to make chained method calls possible.
	 */
	public function setName($p_name)
	{
		if(!is_string($p_name))
		{
			throw new \InvalidArgumentException('$p_name is not a string (' . gettype($p_name) . ')');
		}

		$this->jsonData['name'] = $p_name;

		return $this;
	}


	/**
	 * Updates the creation date of the style configuration
	 *
	 * @param \DateTime $dateTime
	 *
	 * @return ConfigInterface Same instance to make chained method calls possible.
	 */
	public function setCreationDate(\DateTime $dateTime)
	{
		$this->jsonData['creationDate'] = $dateTime->format('Y-m-d H:i:s');

		return $this;
	}


	/**
	 * Updates the modification date of the style configuration
	 *
	 * @param \DateTime $dateTime
	 *
	 * @return \StyleEdit\Entities\Config Same instance to make chained method calls possible.
	 */
	public function setModificationDate(\DateTime $dateTime)
	{
		$this->jsonData['modificationDate'] = $dateTime->format('Y-m-d H:i:s');

		return $this;
	}
	
	
	/**
	 * Updates the boilerplate modification date of the style configuration
	 *
	 * @param \DateTime $dateTime
	 *
	 * @return \StyleEdit\Entities\Config Same instance to make chained method calls possible.
	 */
	public function setBoilerplateModificationDate(\DateTime $dateTime)
	{
		$this->jsonData['boilerplateModificationDate'] = $dateTime->format('Y-m-d H:i:s');
		
		return $this;
	}


	/**
	 * Validates the data structure of a given JSON
	 *
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "name"-element with a non-empty
	 *                                   string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplate"-element with a
	 *                                   non-empty string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "colorPalette"-element
	 *                                   containing an array of string values
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "creationDate"-element with a
	 *                                   formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "modificationDate"-element with
	 *                                   a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplateModificationDate"-
	 *                                   element with a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "settings"-element
	 * @throws \InvalidArgumentException if $jsonData["settings"][]["icon"] is not a string
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "customStyles"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"] does not contain a valid
	 *                                   "entries"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "group"-element if $jsonData["settings"]["type"] is "display"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "position"-element if $jsonData["settings"]["type"] is "boxes"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   boolean value if $jsonData["settings"]["entries"][]["type"] is "switch"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   string value if $jsonData["settings"]["entries"][]["type"] is not "switch" or
	 *                                   "title"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain an
	 *                                   "options" array if $jsonData["settings"]["entries"][]["type"] is "select"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"][]["options"] does not
	 *                                   contain string or array values
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "tags"-element containing an array of strings
	 *
	 * @param array $jsonData
	 */
	protected function _validateJsonDataStructure(array &$jsonData)
	{
		if(!array_key_exists('name', $jsonData) || !is_string($jsonData['name']) || empty($jsonData['name']))
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a "name"-element with a non-empty string value');
		}

		if(!array_key_exists('boilerplate', $jsonData)
		   || !is_string($jsonData['boilerplate'])
		   || empty($jsonData['boilerplate'])
		)
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a "boilerplate"-element with a non-empty string value');
		}

		if(array_key_exists('colorPalette', $jsonData) && !is_array($jsonData['colorPalette']))
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a valid "colorPalette"-element containing an array of string values');
		}

		if(array_key_exists('colorPalette', $jsonData) && is_array($jsonData['colorPalette']))
		{
			foreach($jsonData['colorPalette'] as $colorVariable)
			{
				if(!is_string($colorVariable))
				{
					throw new \InvalidArgumentException('The given array $jsonData does not contain a valid "colorPalette"-element containing an array of string values');
				}
			}
		}

		if(!array_key_exists('creationDate', $jsonData)
		   || preg_match('/[\d]{4}-[\d]{2}-[\d]{2} [\d]{2}:[\d]{2}:[\d]{2}/', $jsonData['creationDate']) !== 1
		)
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a "creationDate"-element with a formatted value "Y-m-d H:i:s"');
		}

		if(!array_key_exists('modificationDate', $jsonData)
		   || preg_match('/[\d]{4}-[\d]{2}-[\d]{2} [\d]{2}:[\d]{2}:[\d]{2}/', $jsonData['modificationDate']) !== 1
		)
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a "modificationDate"-element with a formatted value "Y-m-d H:i:s"');
		}
		
		if(array_key_exists('boilerplateModificationDate', $jsonData) 
		   && preg_match('/[\d]{4}-[\d]{2}-[\d]{2} [\d]{2}:[\d]{2}:[\d]{2}/', 
		                 $jsonData['boilerplateModificationDate']) !== 1
		)
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a "boilerplateModificationDate"-element with a formatted value "Y-m-d H:i:s"');
		}

		if(!array_key_exists('settings', $jsonData)
		   || !is_array($jsonData['settings'])
		)
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a "settings"-element');
		}

		if(!array_key_exists('customStyles', $jsonData) || !is_string($jsonData['customStyles']))
		{
			throw new \InvalidArgumentException('The given array $jsonData does not contain a valid "customStyles"-element');
		}

		foreach($jsonData['settings'] as &$setting)
		{
			if(!is_array($setting)
			   || !array_key_exists('name', $setting)
			   || !array_key_exists('type', $setting)
			   || !array_key_exists('entries', $setting)
			   || !is_string($setting['name'])
			   || !is_string($setting['type'])
			   || !is_array($setting['entries'])
			)
			{
				throw new \InvalidArgumentException('The given array $jsonData does not contain a valid "settings"-element');
			}

			if(array_key_exists('icon', $setting) && !is_string($setting['icon']))
			{
				throw new \InvalidArgumentException('$jsonData["settings"][]["icon"] is not a string');
			}

			foreach($setting['entries'] as &$entry)
			{
				if(!is_array($entry)
				   || !array_key_exists('name', $entry)
				   || !array_key_exists('type', $entry)
				   || !is_string($entry['name'])
				   || !is_string($entry['type'])
				   || ($entry['type'] !== 'title' && !array_key_exists('value', $entry))
				)
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"] does not contain a valid "entries"-element');
				}

				if($entry['type'] === 'text' && !is_string($entry['value']))
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain a valid "text"-element if $jsonData["settings"]["type"] is "text".');
				}
				
				if($entry['type'] === 'image' && (!is_string($entry['value']))) 
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain a valid "image"-element if $jsonData["settings"]["type"] is "image".');
				}
				
				// Old styles might contain relative paths to images but StyleEdit must only store the filename of the
				// image. Update the invalid paths with the base filenames.
				if($entry['type'] === 'image' && strpos($entry['value'], '../../') !== -1)  
				{
					$entry['value'] = \StyleEdit\Utf8Converter::basename($entry['value']);
				}

				if($setting['type'] === 'display'
				   && $entry['type'] !== 'title'
				   && (!array_key_exists('group', $entry)
				       || !is_string($entry['group']))
				)
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain a valid "group"-element if $jsonData["settings"]["type"] is "display"');
				}
				elseif($setting['type'] === 'boxes'
				       && (!array_key_exists('position', $entry)
				           || !is_numeric($entry['position']))
				)
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain a valid "position"-element if $jsonData["settings"]["type"] is "boxes"');
				}

				if($entry['type'] === 'switch' && !is_bool($entry['value']))
				{
					throw new \InvalidArgumentException('$jsonData["settings"]["entries"][]["value"] does not contain a boolean value if $jsonData["settings"]["entries"][]["type"] is "switch"');
				}
				elseif($entry['type'] !== 'switch'
				       && $entry['type'] !== 'title'
				       && !is_string($entry['value'])
				)
				{
					throw new \InvalidArgumentException('$jsonData["settings"]["entries"][]["value"] does not contain a string value if $jsonData["settings"]["entries"][]["type"] is not "switch" or "title"');
				}

				if($entry['type'] === 'select'
				   && (!array_key_exists('options', $entry) || !is_array($entry['options']))
				)
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain an "options" array if $jsonData["settings"]["entries"][]["type"] is "select"');
				}

				if($entry['type'] === 'select')
				{
					foreach($entry['options'] as $option)
					{
						if(!is_string($option) && !is_array($option))
						{
							throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"][]["options"] does not contain string or array values');
						}
					}
				}

				if(array_key_exists('tags', $entry) && !is_array($entry['tags']))
				{
					throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain a valid "tags"-element containing an array of strings');
				}

				if(array_key_exists('tags', $entry))
				{
					foreach($entry['tags'] as $tag)
					{
						if(!is_string($tag))
						{
							throw new \InvalidArgumentException('The given array $jsonData["settings"]["entries"] does not contain a valid "tags"-element containing an array of strings');
						}
					}
				}
			}
		}
	}
}